/*));
 Test/Debug of Octave functionality (on E1433)
 */

#include <stdlib.h>             /* For exit */
#include <stdio.h>              /* For printf */
#include <string.h>             /* For strcmp, strcat */
#include <math.h>               /* For sqrt */
#include "e1432.h"
#include "xplot.h"

#define BA_TIMEOUT 2000000	/* about 1 minute, S700 hpux, mxi */

#define MAX_LAS		2	/* max number of modules */
#define MAX_SRCS	(MAX_LAS * E1432_SRC_CHANS)
#define MAX_CHANS	(MAX_LAS * E1432_INPUT_CHANS)

#define MAX_BLOCKSIZE	16384
#define SAMPLE_FREQ	65536.0

#define PRINT_DB

#define MAX_FRAMESIZE	64

#define MAX_LONG	0x7fffffff

#define SRC_FILTER_FREQ_MAX	25600
#define SRC_FILTER_FREQ_MIN	6400

/* from e1432i.h */
#define E1432_OCTAVE_DEBUG_REG	0x73c
#define E1432_OCTAVE_DEBUG_FLOAT	0x00000004UL
#define E1432_OCTAVE_DEBUG_COUNT	0x00000008UL
#define E1432_OCTAVE_DEBUG_NO_CONVERT	0x00000100UL
#define E1432_OCTAVE_DEBUG_NO_AVG	0x00000200UL
#define E1432_OCTAVE_DEBUG_NO_ANA_CRTN	0x00001000UL
#define E1432_OCTAVE_DEBUG_NO_BAND_CRTN	0x00002000UL
#define E1432_OCTAVE_DEBUG_ALL_BINS	0x00004000UL
#define E1432_OCTAVE_DEBUG_NO_DOWNLOAD	0x00008000UL
#define UNNORMALIZED_ZERO		0x80000000UL
#define UNNORMALIZED_NON_ZERO		0x40000000UL

/* octave band settling times are these numbers / band */
#define FULL_OCT_SETTLING	7.1
#define THIRD_OCT_SETTLING	21.6

#define UNNORMALIZED		(UNNORMALIZED_ZERO | UNNORMALIZED_NON_ZERO)
#define FT_DEFAULT							\
    ( E1432_OCTAVE_DEBUG_FLOAT | E1432_OCTAVE_DEBUG_COUNT		\
    | E1432_OCTAVE_DEBUG_NO_ANA_CRTN | E1432_OCTAVE_DEBUG_NO_BAND_CRTN	\
    | E1432_OCTAVE_DEBUG_ALL_BINS	| E1432_OCTAVE_DEBUG_NO_DOWNLOAD )

/* Wrap this around all the many function calls which might fail */
#define DEBUG(s)        s
#ifdef  __lint
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (func)
#else
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        return _s;\
    }\
} while (0)
#endif


/* enums for command line parm entry */

SHORTSIZ16 octave_mode_enums[] =
{
    E1432_OCTAVE_MODE_FULL, E1432_OCTAVE_MODE_FULL,
    E1432_OCTAVE_MODE_THIRD, E1432_OCTAVE_MODE_THIRD
};
char *octave_mode_strs[] = {
    "F", "f",
    "T", "t",
    NULL
};

SHORTSIZ16 octave_avg_mode_enums[] =
{
    E1432_OCTAVE_AVG_MODE_LIN, E1432_OCTAVE_AVG_MODE_LIN,
    E1432_OCTAVE_AVG_MODE_EXP, E1432_OCTAVE_AVG_MODE_EXP
};
char *octave_avg_mode_strs[] =
{
    "L", "l",
    "E", "e",
    NULL
};

SHORTSIZ16 octave_hold_mode_enums[] =
{
    E1432_OCTAVE_HOLD_MODE_OFF, E1432_OCTAVE_HOLD_MODE_OFF,
    E1432_OCTAVE_HOLD_MODE_MIN, E1432_OCTAVE_HOLD_MODE_MIN,
    E1432_OCTAVE_HOLD_MODE_MIN, E1432_OCTAVE_HOLD_MODE_MIN,
    E1432_OCTAVE_HOLD_MODE_MAX, E1432_OCTAVE_HOLD_MODE_MAX
};
char *octave_hold_mode_strs[] =
{
    "O", "o",
    "M", "m",
    "N", "n",
    "X", "x",
    NULL
};

SHORTSIZ16 data_mode_enums[] =
{
    E1432_BLOCK_MODE,                E1432_BLOCK_MODE,
    E1432_CONTINUOUS_MODE,           E1432_CONTINUOUS_MODE,
    E1432_DATA_MODE_OVERLAP_FREERUN, E1432_DATA_MODE_OVERLAP_FREERUN,
    E1432_DATA_MODE_OVERLAP_BLOCK,   E1432_DATA_MODE_OVERLAP_BLOCK };
char *data_mode_strs[] =
{
    "b", "B",
    "c", "C",
    "f", "F",
    "o", "O",
    NULL
};

SHORTSIZ16 data_size_enums[] =
{
    E1432_DATA_SIZE_16,
    E1432_DATA_SIZE_32_SERV, E1432_DATA_SIZE_32_SERV,
    E1432_DATA_SIZE_FLOAT32, E1432_DATA_SIZE_FLOAT32,
    E1432_DATA_SIZE_32  /* last so will not also match 32s, etc */
};
char *data_size_strs[] =
{
    "16",
    "32s", "32S",
    "f32", "F32",
    "32",
    NULL
};

SHORTSIZ16 weighting_enums[] =
{
    E1432_WEIGHTING_A, E1432_WEIGHTING_A,
    E1432_WEIGHTING_B, E1432_WEIGHTING_B,
    E1432_WEIGHTING_C, E1432_WEIGHTING_C,
    E1432_WEIGHTING_OFF, E1432_WEIGHTING_OFF
};
char *weighting_strs[] =
{
    "a", "A",
    "b", "B",
    "c", "C",
    "o", "O",
    NULL
};

FLOATSIZ64 full_octaves[] =
{
    4.0,    8.0,   16.0,   31.5,   63.0,   125.0, 250.0,
  500.0, 1000.0, 2000.0, 4000.0, 8000.0, 16000.0
};

FLOATSIZ64 third_octaves[] =
{
     3.15,   4.0,    5.0,     6.3,     8.0,    10.0,    12.5,   16.0,
    20.0,   25.0,   31.5,    40.0,    50.0,    63.0,    80.0,  100.0,
   125.0,  160.0,  200.0,   250.0,   315.0,   400.0,   500.0,  630.0,
   800.0, 1000.0, 1250.0,  1600.0,  2000.0,  2500.0,  3150.0, 4000.0,
  5000.0, 6300.0, 8000.0, 10000.0, 12500.0, 16000.0, 20000.0
};

const char reg_meas_err[] =
  "\nOctave regression test error on chan %d%s, %.3g Hz band%s\n  read %lg dB";


#define VALID_OPTS	\
  "a:bc:e:f:gh:i:l:m:no:p:r:s:t:uvw:x:y:z:A:C:D:E:F:L:M:N:OQ:RS:TU:VW:XYZ"


void
print_usage(char *name)
{
#   if E1432_OCTAVE_MODE_DEF == E1432_OCTAVE_MODE_FULL
        char md = 'F';
#   else
        char md = 'T';
#   endif

#   if E1432_OCTAVE_AVG_MODE_DEF == E1432_OCTAVE_AVG_MODE_LIN
        char ad = 'L';
#   else
        char ad = 'E';
#   endif

#   if E1432_OCTAVE_HOLD_MODE_DEF == E1432_OCTAVE_HOLD_MODE_MAX
        char hd = 'X';
#   endif
#   if E1432_OCTAVE_HOLD_MODE_DEF == E1432_OCTAVE_HOLD_MODE_MIN
	char hd = 'N';
#   endif
#   if E1432_OCTAVE_HOLD_MODE_DEF == E1432_OCTAVE_HOLD_MODE_OFF
        char hd = 'O';
#   endif

    (void) fprintf(stdout,"usage: %s [options] file\n",name);
    (void) fprintf(stdout,"options:\n");
    (void) fprintf(stdout,"    -a octave_avg_mode [Lin, Exp]"
      " (default %c)\n", ad);
    (void) fprintf(stdout,"    -b (pause between Octave data reads)\n");
    (void) fprintf(stdout,"    -c octave_time_const"
      " (default %g)\n", E1432_OCTAVE_TIME_CONST_DEF);
    (void) fprintf(stdout,"    -e max_frame_test_errors\n");
    (void) fprintf(stdout,"    -f filter_settling_time"
      " (default %g)\n", E1432_FILTER_SETTLING_DEF);
    (void) fprintf(stdout,"    -g (read gap)\n");
    (void) fprintf(stdout,"    -h octave_hold_mode [maX, miN, Off]"
      " (default %c)\n", hd);
    (void) fprintf(stdout,"    -i octave_int_time"
      " (default %g)\n", E1432_OCTAVE_INT_TIME_DEF);
    (void) fprintf(stdout,"    -l time blocksize"
      " (default %g)\n", E1432_BLOCKSIZE_DEF);
    (void) fprintf(stdout,"    -m loops per measurement restart\n");
    (void) fprintf(stdout,"    -n (no warnings)\n");
    (void) fprintf(stdout,"    -o octave_mode [Full, Third]"
      " (default %c)\n", md);
    (void) fprintf(stdout,"    -r range\n");
    (void) fprintf(stdout,"    -s data_size [16, 32, 32S, F32]\n");
    (void) fprintf(stdout,"    -t octave_time_step"
      " (default %g)\n", E1432_OCTAVE_TIME_STEP_DEF);
    (void) fprintf(stdout,"    -v (verbose)\n");
    (void) fprintf(stdout,"    -w source frequency (default 0)\n");
    (void) fprintf(stdout,"    -x source amplitude (default 0)\n");
    (void) fprintf(stdout,"    -y number of FFT measurements\n");
    (void) fprintf(stdout,"    -z channel to cycle between"
      " (need -z twice, min, plus -m opt)\n");
    (void) fprintf(stdout,"    -A coupling_freq (default none)\n");
    (void) fprintf(stdout,"    -C channel (default all)\n");
    (void) fprintf(stdout,"    -D debug_mode\n");
    (void) fprintf(stdout,"    -E octave_stop_freq"
      " (default %g)\n", E1432_OCTAVE_STOP_FREQ_DEF);
    (void) fprintf(stdout,"    -F freq (to read freq data)\n");
    (void) fprintf(stdout,"    -L logical_address (default 8)\n");
    (void) fprintf(stdout,"    -M data_mode (b, o, f, or c)\n");
    (void) fprintf(stdout,"    -N loops\n");
    (void) fprintf(stdout,"    -O print overloads\n");
    (void) fprintf(stdout,"    -Q num_averages\n");
    (void) fprintf(stdout,"    -R read peak/rms\n");
    (void) fprintf(stdout,"    -S octave_start_freq"
      " (default %g)\n", E1432_OCTAVE_START_FREQ_DEF);
    (void) fprintf(stdout,"    -T read time data also\n");
    (void) fprintf(stdout,"    -V read current octave data\n");
    (void) fprintf(stdout,"    -W weighting (A, B, C, or O) (default O)\n");
    (void) fprintf(stdout,"    -Z (get raw data)\n");
}


void
parm_err(char *arg, char *parm_name)
{
    (void) fprintf(stderr, "parameter error: %s \"%s\" not converted\n",
      parm_name, arg);
    exit(2);
}


LONGSIZ32
get_long(char *arg, char *parm_name)
{
    FLOATSIZ32 ftmp;
    LONGSIZ32 lval = 0;
    int rtn;

    /* strip leading "0x", if there */
    if ( strncmp(arg, "0x", 2) == 0 )
    {
        arg += 2;
        rtn = sscanf(arg, "%x", &lval);
    }
    else /* try float => long conversion next */
    {
        rtn = sscanf(arg, "%f", &ftmp);
        if ( rtn == 1 )
        {
	    lval = (LONGSIZ32)(ftmp + .5);
        }
	else /* try hex as a last ditch effort */
	{
            rtn = sscanf(arg, "%x", &lval);
	}
    }
    if ( rtn != 1 ) parm_err(arg, parm_name);

    return lval;
}


SHORTSIZ16
get_short(char *arg, char *parm_name)
{
    LONGSIZ32 ltmp = get_long(arg, parm_name);
    return (SHORTSIZ16)ltmp;
}


FLOATSIZ32
get_float(char *arg, char *parm_name)
{
    FLOATSIZ32 ftmp;
    int rtn = sscanf(optarg, "%f", &ftmp);

    if ( rtn != 1 ) parm_err(arg, parm_name);
    return ftmp;
}


SHORTSIZ16
get_enum(char *arg, SHORTSIZ16 *enums, char **strs, char *parm_name)
{
    while ( *strs != NULL )
    {
        if ( strcmp(arg, *strs++) == 0 ) return *enums;
	enums++;
    }
    parm_err(arg, parm_name);
}


void prompt(char* promptstr)
{
    char c;
    fprintf(stderr, promptstr);
    c = getchar();
    if ( c == 'q' || c == 'Q' ) exit(0);
}


SHORTSIZ16 print_lims(E1432ID hw, SHORTSIZ16 group,
 SHORTSIZ16 (*plimf)(E1432ID, SHORTSIZ16, FLOATSIZ32 *, FLOATSIZ32 *,
 FLOATSIZ32 *, FLOATSIZ32 *), char *pname)
{
    FLOATSIZ32 min, max, def, step;
    SHORTSIZ16 err;

    err = (*plimf)(hw, group, &min, &max, &def, &step);

    if ( err )
    {
	(void) printf("encountered error number %d with e1432_get_%s_limits\n",
	  err, pname);
    }
    else
    {
	(void) printf("%s min = %g, max = %g, def = %g, step = %g\n",
	  pname, min, max, def, step);
    }
    return err;
}


SHORTSIZ16 print_parms(E1432ID hw, SHORTSIZ16 group)
{
    SHORTSIZ16 err = 0;
    int i;
    char *oct_str, *avg_str, *hold_str, *dm;
    SHORTSIZ16 data_mode;
    SHORTSIZ16 octave_mode, octave_avg_mode, octave_hold_mode;
    LONGSIZ32 octave_blocksize;
    FLOATSIZ32 octave_start_freq, octave_stop_freq;
    FLOATSIZ32 octave_int_time, octave_time_const, octave_time_step;

    CHECK(e1432_get_data_mode(hw, group, &data_mode));
    for ( i = 0; data_mode_enums[i] != data_mode ; i++ );
    dm = data_mode_strs[i];
    (void) printf("data_mode = %s\n", dm);

    CHECK(e1432_get_octave_mode(hw, group, &octave_mode));
    if ( octave_mode == E1432_OCTAVE_MODE_FULL ) oct_str = "full";
    else if ( octave_mode == E1432_OCTAVE_MODE_THIRD ) oct_str = "third";
    else oct_str = "undefined";
    CHECK(e1432_get_octave_avg_mode(hw, group, &octave_avg_mode));
    if ( octave_avg_mode == E1432_OCTAVE_AVG_MODE_LIN ) avg_str = "lin";
    else if ( octave_avg_mode == E1432_OCTAVE_AVG_MODE_EXP ) avg_str ="exp";
    else avg_str = "undefined";
    CHECK(e1432_get_octave_hold_mode(hw, group, &octave_hold_mode));
    if ( octave_hold_mode == E1432_OCTAVE_HOLD_MODE_OFF ) hold_str = "off";
    else if ( octave_hold_mode == E1432_OCTAVE_HOLD_MODE_MAX ) hold_str = "max";
    else if ( octave_hold_mode == E1432_OCTAVE_HOLD_MODE_MIN ) hold_str = "min";
    else hold_str = "undefined";
    (void) printf("octave_mode = %s, octave_avg_mode = %s,"
      " octave_hold_mode = %s\n", oct_str, avg_str, hold_str);

    CHECK(e1432_get_octave_start_freq(hw, group, &octave_start_freq));
    CHECK(e1432_get_octave_stop_freq(hw, group, &octave_stop_freq));
    (void) printf("octave_start_freq = %g, octave_stop_freq = %g\n",
      octave_start_freq, octave_stop_freq);
    print_lims(hw, group, e1432_get_octave_start_freq_limits,
      "octave_start_freq");
    print_lims(hw, group, e1432_get_octave_stop_freq_limits,
      "octave_stop_freq");

    CHECK(e1432_get_octave_int_time(hw, group, &octave_int_time));
    (void) printf("octave_int_time = %g\n", octave_int_time);
    print_lims(hw, group, e1432_get_octave_int_time_limits,
      "octave_int_time");

    CHECK(e1432_get_octave_time_const(hw, group, &octave_time_const));
    CHECK(e1432_get_octave_time_step(hw, group, &octave_time_step));
    (void) printf("octave_time_const = %g, octave_time_step %g\n",
      octave_time_const, octave_time_step);
    print_lims(hw, group, e1432_get_octave_time_const_limits,
      "octave_time_const");
    print_lims(hw, group, e1432_get_octave_time_step_limits,
      "octave_time_step");


    (void) printf("\n");
    return err;
}


wait_block_available(E1432ID hw, SHORTSIZ16 group, LONGSIZ32 loop)
{
    SHORTSIZ16 status;
    char *error_string;
    LONGSIZ32 ba_count = 0;

    do  /* Wait for block available and check for errors and warnings  */
    {
        status = e1432_block_available(hw, group);
        error_string = e1432_get_error_string();
    } while ( status != 1 && ba_count++ < BA_TIMEOUT
      && *error_string == NULL ); 
    if ( status < 0 || *error_string != NULL )
    {
        (void) fprintf(stderr, "block_available() error \"%s\", loop %d\n",
          error_string, loop);
	if ( status < 0 ) return status;
        return -1;
    }
    if ( ba_count >= BA_TIMEOUT )
    {
        (void) fprintf(stderr, "block_available() timeout, loop %d\n",
          loop);
        return -1;
    }
    return 0;
}


SHORTSIZ16
get_debug_reg(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 *debug_reg)
{
    CHECK(e1432_read32_register(hw, ID, E1432_OCTAVE_DEBUG_REG, debug_reg));
    /* clear the two msbs and write back */
    CHECK(e1432_write32_register(hw, ID, E1432_OCTAVE_DEBUG_REG,
      *debug_reg & 0x3fffffff));
    return 0;
}


SHORTSIZ16
print_peak_rms(E1432ID hw, SHORTSIZ16 ID, struct e1432_trailer *trailer)
{
    float curr_peak, curr_rms;

    printf("  trailer Peak = %g, RMS = %g\n", trailer->peak, trailer->rms);
    CHECK(e1432_get_current_value(hw, ID, E1432_CURRENT_VAL_PEAK, &curr_peak));
    CHECK(e1432_get_current_value(hw, ID, E1432_CURRENT_VAL_RMS, &curr_rms));
    printf("  current Peak = %g, RMS = %g\n", curr_peak, curr_rms);

    return 0;
}


int oct_band(FLOATSIZ64 freq, SHORTSIZ16 octave_mode,
  FLOATSIZ32 octave_start_freq)
{
    FLOATSIZ64 *band;
    FLOATSIZ64 half_bandwidth;
    FLOATSIZ64 tmp;
    int i = 0;

    if ( octave_mode == E1432_OCTAVE_MODE_THIRD )
    {
	    half_bandwidth = pow(2.0, 1.0/6.0);
	    band = &third_octaves[0];
    }
    else
    {
	    half_bandwidth = sqrt(2.0);
	    band = &full_octaves[0];
    }

    tmp = octave_start_freq / half_bandwidth;
    while ( *band < tmp) band++;
    tmp = freq / half_bandwidth;
    while ( *band < tmp)
    {
	i++;
        band++;
    }
    return i;
}


/* averaging */
FLOATSIZ64 freq_avg[MAX_CHANS];
FLOATSIZ64 octave_avg[MAX_CHANS][MAX_FRAMESIZE];

void
zero_avg_buffs(void)
{
    int c, i;
    for ( c = 0; c < MAX_CHANS; c++)
    {
        freq_avg[c] = 0.0;
	for ( i = 0; i < MAX_FRAMESIZE; i++ )
	{
            octave_avg[c][i] = 0.0;
	}
    }
}


FLOATSIZ64 dB(FLOATSIZ64 pwr)
{
    return 10.0 * log10(pwr);
}


int
main(int argn, char **argv)
{
    /* command line vars */
    int opt;               /* really char returned */

    /* state vars */
    int debug_pause = 0;
    int get_raw = 0;
    int frame_test = 0;
    int meas_restart = 0;
    int read_time = 0;
    int print_time = 0;
    int read_freq = 0;
    int read_current = 0;
    int read_trailer = 0;
    int do_peak_rms_print = 0;
    int print_current = 0;
    int print_oct = 1;
    int print_ovld = 0;
    int verbose = 0;
    int warnings = 1; /* default to print warnings */
    int regression_test = 0;
    LONGSIZ32 loops = -1;
    int spec_chans = 0;
    SHORTSIZ16 spec_chan_list[MAX_CHANS];
    int cycle_chans = 0;
    int cycle_chan = 0;
    SHORTSIZ16 cycle_chan_list[MAX_CHANS];
    SHORTSIZ16 cycle_in_group[MAX_CHANS];
    LONGSIZ32 fft_meas = 0;

    /* boiler plate vars */
    SHORTSIZ16 modules = 1;
    int la_override = 0;
    SHORTSIZ16 laddr[MAX_LAS] = { 8 };	/* default logical address */
    E1432ID hw;
    struct e1432_hwconfig hwconfig[MAX_LAS];
    SHORTSIZ16 in_chans = 0;
    SHORTSIZ16 in_chan_list[MAX_CHANS];
    SHORTSIZ16 first_chan;
    SHORTSIZ16 in_group;

    /* general use */
    int i, j;
    SHORTSIZ16 status, error;
    LONGSIZ32 loop;
    LONGSIZ32 ba_count;
    FLOATSIZ64 dTmp;

    /* general api vars */
    LONGSIZ32 blocksize = E1432_BLOCKSIZE_DEF;
    SHORTSIZ16 data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
    SHORTSIZ16 data_size = E1432_DATA_SIZE_DEF;
    SHORTSIZ16 weighting = E1432_WEIGHTING_OFF;
    FLOATSIZ32 range = 1.0;
    SHORTSIZ16 coupling = E1432_COUPLING_DC;
    FLOATSIZ32 coupling_freq = 1.0;
    FLOATSIZ32 clock_freq = SAMPLE_FREQ;

    /* time data */
    SHORTSIZ16 time_enable;
    FLOATSIZ64 time_data[MAX_BLOCKSIZE];

    /* freq data */
    SHORTSIZ16 freq_enable;
    FLOATSIZ64 freq_data[MAX_BLOCKSIZE];
    FLOATSIZ64 test_freq;
    FLOATSIZ64 *rePtr, *imPtr;
    FLOATSIZ64 bin_space;
    FLOATSIZ64 freq_raw;
    int test_freq_index;

    /* overloads */
    SHORTSIZ16 ovld_tlr_any[MAX_CHANS];
    SHORTSIZ16 ovld_tlr_com[MAX_CHANS];
    SHORTSIZ16 ovld_chk_any[MAX_CHANS];
    SHORTSIZ16 ovld_chk_com[MAX_CHANS];
    SHORTSIZ16 ovld_chk_dif[MAX_CHANS];
    SHORTSIZ16 ovld_chk_half[MAX_CHANS];
    int overload;

    /* gap */
    LONGSIZ32 prev_gap;
    LONGSIZ32 xpect_gap;
    int read_gap = 0;

    /* plot */
    int plot = 0;
    SHORTSIZ16 plot_chan = -1;
    char *plotid = NULL;
    float plot_data[MAX_BLOCKSIZE];

    /* octave data */
    LONGSIZ32 octave_raw_data[MAX_BLOCKSIZE];
    FLOATSIZ64 octave_data[MAX_CHANS][MAX_FRAMESIZE];
    FLOATSIZ64 *oct_data[MAX_CHANS];
    struct e1432_trailer trailer;
    LONGSIZ32 count;

    /* octave api vars */
    SHORTSIZ16 octave_mode = E1432_OCTAVE_MODE_DEF;
    SHORTSIZ16 octave_avg_mode = E1432_OCTAVE_AVG_MODE_DEF;
    SHORTSIZ16 octave_hold_mode = E1432_OCTAVE_HOLD_MODE_DEF;
    FLOATSIZ32 octave_start_freq = E1432_OCTAVE_START_FREQ_DEF;
    FLOATSIZ32 octave_stop_freq = E1432_OCTAVE_STOP_FREQ_DEF;
    FLOATSIZ32 octave_int_time = E1432_OCTAVE_INT_TIME_DEF;
    FLOATSIZ32 octave_time_const = E1432_OCTAVE_TIME_CONST_DEF;
    FLOATSIZ32 octave_time_step = E1432_OCTAVE_TIME_STEP_DEF;
    FLOATSIZ32 filter_settling_time = E1432_FILTER_SETTLING_DEF;

    /* derived octave vars */
    LONGSIZ32 octave_blocksize;
    int data_mult = 1;  /* 1 for 32 bit data, 2 for double data */
    int lin_avg;
    int frame_length = 64;

    /* source */
    int src_on = 0;
    FLOATSIZ32 src_freq = 0.0;
    FLOATSIZ32 src_amp = 0.0;
    SHORTSIZ16 src_chan_list[MAX_SRCS];
    SHORTSIZ16 src_group;
    int src_chans;

    /* averaging */
    LONGSIZ32 num_averages = 0;

    /* frame test vars */
    int avgs, upds, frame;
    long update;
    int prev_first_raw_val = 0;
    int ft_errs = 0;
    LONGSIZ32 max_ft_errs = MAX_LONG;

    /* regression test vars*/
    int reg_errs = 0;
    FLOATSIZ64 band_step, half_bandwidth, flat_bandwidth, trans_bandwidth;
    FLOATSIZ64 *oct_table;
    FLOATSIZ64 band_settle_factor;
    FLOATSIZ64 low_band_ignore; /* low bands still settling - ignore */
    FLOATSIZ64 update_rate;
    FLOATSIZ64 elapsed_time;
    int reg_sig_chan[MAX_CHANS];

    /* Regression Test Assumptions: */
    /* Sine input 452 Hz, 1.5 Vp */
    FLOATSIZ64 reg_sine_freq = 452;
    FLOATSIZ64 reg_sine_amp = 1.5;
    /* input in chans 1,2,4,6 (none in 3,5,8) (source in 7) */
    const int reg_sig_chan_list[] = {1, 2, 4, 6};
    const int reg_sig_chans = sizeof(reg_sig_chan_list)/sizeof(int);

    /* Regression Test limits */
    int reg_errs_max = 10;	/* maximum errors printed */
    FLOATSIZ64 off_chan_down = 1.e-5; /* signal in disconnected chans */
    FLOATSIZ64 out_of_band = 1.e-6; /* out of and response */
#define IN_BAND_LIM	1.2589	/* in band limits +- 1 dB */
    FLOATSIZ64 in_band_max = IN_BAND_LIM;
    FLOATSIZ64 in_band_min = 1.0/IN_BAND_LIM;
    FLOATSIZ64 band_edge_min = .5;    /* in band but skirt, allow 3 dB */
    FLOATSIZ64 trans_bands = 4; /* transition bands - not tested */
#define PK_RMS_LIM	1.0023

    /* debug */
    LONGSIZ32 debug_mode = 0;
    LONGSIZ32 debug_reg;


    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);

    while ( ( opt = getopt(argn, argv, VALID_OPTS) ) != EOF )
    {
        switch (opt)
        {
        case 'a':
            octave_avg_mode = get_enum(optarg, octave_avg_mode_enums,
	      octave_avg_mode_strs, "octave_avg_mode");
            break;
        case 'b':
            debug_pause = 1;
            break;
        case 'c':
	    octave_time_const = get_float(optarg, "octave_time_const");
            break;
        case 'e':
            max_ft_errs = get_long(optarg, "max frame test errors");
            break;
        case 'f':
	    filter_settling_time = get_float(optarg, "filter_settling_time");
            break;
        case 'g':
            read_gap = 1;
            break;
        case 'h':
            octave_hold_mode = get_enum(optarg, octave_hold_mode_enums,
	      octave_hold_mode_strs, "octave_hold_mode");
            break;
        case 'i':
	    octave_int_time = get_float(optarg, "octave_int_time");
            break;
        case 'l':
            blocksize = get_long(optarg, "time blocksize");
            break;
        case 'm':
            meas_restart = get_long(optarg, "meas restart");
            break;
        case 'n':
            warnings = 0; /* turn off warnings */
            break;
        case 'o':
            octave_mode = get_enum(optarg, octave_mode_enums, octave_mode_strs,
	      "octave_mode");
            break;
        case 'p':
            plot = 1;
	    plot_chan = get_short(optarg, "channel to plot");
            break;
        case 'r':
            range = get_float(optarg, "range");
            break;
        case 's':
            data_size =
	      get_enum(optarg, data_size_enums, data_size_strs, "data size");
            break;
        case 't':
	    octave_time_step = get_float(optarg, "octave_time_step");
            break;
        case 'v':
            verbose = 1;
            break;
        case 'w':
            src_freq = get_float(optarg, "source frequency");
            src_on = 1;
            break;
        case 'x':
            src_amp = get_float(optarg, "source amplitude");
            src_on = 1;
            break;
        case 'y':
            fft_meas = get_long(optarg, "number of FFT measurements");
            break;
        case 'z':
	    if ( cycle_chans < MAX_CHANS - 1 )
	    {
		cycle_chan_list[cycle_chans++] =
	          get_short(optarg, "cycle channel");
	    }
	    else
	    {
                (void) fprintf(stderr, "Number of cycle channels exceeds %d\n",
		  MAX_CHANS);
                exit(2);
	    }
            break;
        case 'A':
            coupling = E1432_COUPLING_AC;
	    coupling_freq = get_float(optarg, "coupling frequency");
            break;
        case 'C':
	    spec_chan_list[spec_chans++] =
	      get_short(optarg, "measurement channel");
            break;
        case 'D':
            debug_mode = get_long(optarg, "debug mode");
            break;
        case 'E':
	    octave_stop_freq = get_float(optarg, "octave_stop_freq");
            break;
        case 'F':
            test_freq = get_float(optarg, "frequency");
            read_freq = 1;
            break;
        case 'L':
	    if ( ! la_override )
	    {
	        la_override = 1;
	        modules = 0;
	    }
	    laddr[modules++] = get_short(optarg, "logical address");
            break;
        case 'M':
            data_mode =
	      get_enum(optarg, data_mode_enums, data_mode_strs, "data mode");
            break;
        case 'N':
            loops = get_long(optarg, "read data loops");
            break;
        case 'O':
	    print_ovld = 1;
            break;
        case 'Q':
            num_averages = get_long(optarg, "num_averages");
            break;
        case 'R':
            do_peak_rms_print = 1;
            break;
        case 'S':
	    octave_start_freq = get_float(optarg, "octave_start_freq");
            break;
        case 'T':
            read_time = 1;
            print_time = 1;
            break;
        case 'U':
	    clock_freq = get_float(optarg, "clock/sample frequency");
            break;
        case 'V':
	    read_current = 1;
            break;
        case 'W':
            weighting = get_enum(optarg, weighting_enums,
	      weighting_strs, "weighting");
            break;
        case 'X':
            frame_test = 1;
            break;
        case 'Y':
	    /* See Regression Test Assumptions above */
            regression_test = 1;
            break;
        case 'Z':
            get_raw = 1;
            break;
        default:
	    print_usage(argv[0]);
            exit(2);
        }
    }

    print_current = read_current;
    if ( do_peak_rms_print || print_ovld || read_gap ) read_trailer = 1;

    lin_avg = (octave_avg_mode == E1432_OCTAVE_AVG_MODE_LIN);
    time_enable = read_time ? E1432_ENABLE_ON : E1432_ENABLE_OFF;
    freq_enable = read_freq ? E1432_ENABLE_ON : E1432_ENABLE_OFF;

    bin_space = clock_freq/(double)blocksize;

    if ( read_freq )
    {
        int bin_index;

        bin_index = (int)(test_freq/bin_space + .5);
        rePtr = &freq_data[2*bin_index];
        imPtr = rePtr + 1;

        test_freq_index = oct_band(test_freq, octave_mode, octave_start_freq);
    }

    if ( num_averages > 0 )
    {
        print_oct = 0;  /* averaging will print instead */
        print_time = 0; /* don't print them */
    }

    if ( frame_test || regression_test )
    {
	/* only print out errors */
        print_oct = 0;
        print_current = 0;
        do_peak_rms_print = 0;
    }

    if ( frame_test )
    {
	/* framing test counts on real, double data, FT_DEFAULT debug_mode */
	get_raw = 0;
	if ( debug_mode == 0 ) debug_mode = FT_DEFAULT;
        filter_settling_time = 0.0;  /* no data ignored */
    }

    if ( regression_test )
    {
        range = 2.0;
        if ( loops == -1 ) loops = 10;  /* 10 loops unless otherwise spec'd */
	if ( filter_settling_time == E1432_FILTER_SETTLING_DEF )
	{
            filter_settling_time = 1.0;  /* ignore 56k settling */
            if ( lin_avg )
            {
                filter_settling_time = 0;
            }
            else
            {
                filter_settling_time = 3 * octave_time_const;
            }
	}
	if ( octave_mode == E1432_OCTAVE_MODE_THIRD )
	{
	    band_step = pow(2.0, 1.0/3.0);
	    oct_table = &third_octaves[0];
            band_settle_factor = THIRD_OCT_SETTLING;
	}
	else
	{
	    band_step = 2.0;
	    oct_table = &full_octaves[0];
            band_settle_factor = FULL_OCT_SETTLING;
	}
        half_bandwidth = sqrt(band_step);
	flat_bandwidth = pow(half_bandwidth, 2.0/3.0);
        trans_bandwidth = pow(band_step, trans_bands);
    }

    CHECK(e1432_assign_channel_numbers(modules, laddr, &hw));

    CHECK(e1432_get_hwconfig(modules, laddr, hwconfig));

    for ( i = 0; i < modules; i++ )
    {
        if ( ! hwconfig[i].oct_present)
        {
	    if ( warnings )
	    {
                (void) fprintf(stderr, "Octave option not present at LA %d\n",
	          laddr[i]);
	    }
	    exit(1);
        }
        in_chans += hwconfig[i].input_chans;
	src_chans += hwconfig[i].source_chans;
    }

    /* set up in_chan_list, in_group */
    if ( in_chans > MAX_CHANS )
    {
	if ( warnings )
	{
            (void) fprintf(stderr,
	      "Warning: %d channels found, only %d will be used\n",
	      in_chans, MAX_CHANS);
	}
	in_chans = MAX_CHANS;
    }
    else if ( in_chans <= 0 )
    {
        (void) fprintf(stderr, "Warning: no input channels found\n");
    }
    else if ( verbose )  printf("%d channels\n", in_chans);

    /* Create channel group */
    {
        int tmpChan = 0;
        int found;
        for (i = 0; i < in_chans; i++)
        {
            if ( spec_chans <= 0 )
            {
                found = 1;
            }
            else
            {
                found = 0;
                for (j = 0; j < spec_chans; j++ )
                {
                    if ( E1432_INPUT_CHAN(i+1) == spec_chan_list[j] ) found = 1;
                }
            }
            if ( found )
            {
                in_chan_list[tmpChan] = E1432_INPUT_CHAN(i+1);
                oct_data[tmpChan] = &octave_data[tmpChan][0];;
		/* determine whether this channel has regression test input */
                reg_sig_chan[tmpChan] = 0;
		for ( j = 0; j < reg_sig_chans; j++ )
		{
		    if ( in_chan_list[tmpChan] == reg_sig_chan_list[j] )
		    {
                        reg_sig_chan[tmpChan] = 1;
		        break;
		    }
		}
                tmpChan++;
            }
        }
        in_chans = tmpChan;
        in_group = e1432_create_channel_group(hw, in_chans, in_chan_list);
        if (in_group >= 0)
        {
            DEBUG((void) printf("e1432_create_channel_group in_group"
	      " returned %d\n", in_group));
            return -1;
        }
    }
    first_chan = in_chan_list[0];

    /* create source group, if needed and found */
    if ( src_on )
    {
	if ( src_chans < 1 )
	{
	    if ( warnings )
	    {
                (void) fprintf(stderr, "Warning, no source channels found\n");
	    }
            src_on = 0;
	}
	else
	{
	    if ( src_chans > MAX_SRCS )
	    {
	        if ( warnings )
	        {
	            (void) fprintf(stderr, "Warning: %d source channels found,"
		      " only %d will be used\n", src_chans, MAX_SRCS);
	        }
		src_chans = MAX_SRCS;
	    }
            for (i = 0; i < src_chans; i++)
	    {
                src_chan_list[i] =  E1432_SOURCE_CHAN(i + 1);
	    }
            src_group =
	      e1432_create_channel_group(hw, src_chans, src_chan_list);
            if ( src_group >= 0 )
            {
                (void) fprintf(stderr, "e1432_create_channel_group in_group"
		  " return %d\n", src_group);
                return -1;
            }
	}
    }

    /* Initialize general hardware things */
    CHECK(e1432_set_analog_input(hw, in_group, E1432_INPUT_MODE_VOLT,
      E1432_INPUT_HIGH_NORMAL, E1432_ANTI_ALIAS_ANALOG_ON,
      E1432_COUPLING_DC, range));
    CHECK(e1432_set_weighting(hw, in_group, weighting));
    CHECK(e1432_set_coupling(hw, in_group, coupling));
    CHECK(e1432_set_coupling_freq(hw, in_group, coupling_freq));
    CHECK(e1432_set_data_size(hw, in_group, data_size));
    CHECK(e1432_set_blocksize(hw, in_group, blocksize)); 
    CHECK(e1432_set_clock_freq(hw, in_group, clock_freq));
    CHECK(e1432_set_span(hw, in_group, clock_freq/2.56));
    CHECK(e1432_set_data_mode(hw, in_group, data_mode));

    if ( read_freq )
    {
        CHECK(e1432_set_calc_data(hw, in_group, E1432_DATA_FREQ));
    }

    if ( read_trailer )
    {
        CHECK(e1432_set_append_status(hw, in_group, E1432_APPEND_STATUS_ON));
    }

    /* setup source, if on */
    if ( src_on )
    {
	FLOATSIZ32 src_range = range;
	FLOATSIZ32 filter_freq = (src_freq < .8 * SRC_FILTER_FREQ_MIN) ?
	  SRC_FILTER_FREQ_MIN : SRC_FILTER_FREQ_MAX;
        CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));
        CHECK(e1432_set_range(hw, src_group, src_range));
        CHECK(e1432_set_amp_scale(hw, src_group, src_amp/src_range));
        CHECK(e1432_set_filter_freq(hw, src_group, filter_freq));
        CHECK(e1432_set_sine_freq(hw, src_group, src_freq));
	CHECK(e1432_set_ramp_rate(hw, src_group, 0.0));
    }

    /* octave setup */
    /* first, set the Octave debug register */
    CHECK(e1432_write32_register(hw, first_chan, E1432_OCTAVE_DEBUG_REG,
      debug_mode));

    CHECK(e1432_set_octave_meas(hw, in_group, E1432_OCTAVE_MEAS_ON));
    CHECK(e1432_set_octave_mode(hw, in_group, octave_mode));
    CHECK(e1432_set_octave_avg_mode(hw, in_group, octave_avg_mode));
    CHECK(e1432_set_octave_hold_mode(hw, in_group, octave_hold_mode));
    CHECK(e1432_set_octave_start_freq(hw, in_group, octave_start_freq));
    CHECK(e1432_set_octave_stop_freq(hw, in_group, octave_stop_freq));
    CHECK(e1432_set_octave_int_time(hw, in_group, octave_int_time));
    CHECK(e1432_set_octave_time_const(hw, in_group, octave_time_const));
    CHECK(e1432_set_octave_time_step(hw, in_group, octave_time_step));
    CHECK(e1432_set_filter_settling_time(hw, in_group, filter_settling_time));

    /* pick up the actual values */
    CHECK(e1432_get_octave_avg_mode(hw, in_group, &octave_avg_mode));
    CHECK(e1432_get_octave_hold_mode(hw, in_group, &octave_hold_mode));
    CHECK(e1432_get_octave_start_freq(hw, in_group, &octave_start_freq));
    CHECK(e1432_get_octave_stop_freq(hw, in_group, &octave_stop_freq));
    CHECK(e1432_get_octave_int_time(hw, in_group, &octave_int_time));
    CHECK(e1432_get_octave_time_const(hw, in_group, &octave_time_const));
    CHECK(e1432_get_octave_time_step(hw, in_group, &octave_time_step));
    CHECK(e1432_get_filter_settling_time(hw, in_group, &filter_settling_time));

    update_rate = octave_time_step;
    if ( lin_avg && octave_int_time < update_rate )
    {
        update_rate = octave_int_time;
    }
    elapsed_time = filter_settling_time;
    if ( elapsed_time < 1.0/512.0 ) elapsed_time = 1.0/512.0;
    xpect_gap = clock_freq * update_rate + .5;

    /* turn on/off data */
    CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_OCTAVE,
      E1432_ENABLE_ON));
    CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_TIME, time_enable));
    CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_FREQ, freq_enable));

    if ( verbose ) print_parms(hw, in_group);

    if ( frame_test )
    {
	float favgs = .5 + octave_int_time * 512.0;
	float fupds = .5 + octave_time_step * 512.0;
        if ( lin_avg )
	{
            avgs = favgs;
	    if ( octave_time_step > octave_int_time ) upds = avgs;
	    else upds = fupds;
	}
	else
	{
	    avgs = MAX_LONG;
	    upds = fupds;
	}
	frame = 0;
    }
    if ( plot )
    {
        plotid = xplot_init_plot(plot_data, blocksize,
	  (float)blocksize/clock_freq, range, -range, TIME_TRACE,
	  "500x400+0+0", "Time Data");
      xplot_change_yautoscale(plotid, 0);
      xplot_set_xscale(plotid, (float)0.0, (float)blocksize/clock_freq);
      xplot_set_yscale(plotid, 2.0*range, -2.0*range);
      xplot_change_xlabel(plotid, "Seconds");
      xplot_change_ylabel(plotid, "Volts");
      xplot_repaint(plotid); 
    }

    zero_avg_buffs();

    /* Start measurement */
    CHECK(e1432_init_measure(hw, in_group));

    /* must be done after measurement starts, since computed at start */
    CHECK(e1432_get_octave_blocksize(hw, in_group, &octave_blocksize));

    /* clear the debug_reg msbs */
    if ( warnings ) CHECK(get_debug_reg(hw, first_chan, &debug_reg));

    for ( loop = 0; loop != loops && ft_errs < max_ft_errs; loop++ )
    {
	if ( wait_block_available(hw, in_group, loop) < 0 )
	{
            ft_errs++;
	    break;
	}

	if ( warnings )
	{
            CHECK(get_debug_reg(hw, first_chan, &debug_reg));
	    if ( debug_reg & UNNORMALIZED )
	    {
                (void)printf("unnormalized ");
	        if ( debug_reg & UNNORMALIZED_ZERO )
		{
	            if ( debug_reg & UNNORMALIZED_NON_ZERO )
		    {
                        (void)printf("zero and non-zero");
		    }
		    else
		    {
                        (void)printf("zero");
		    }
		}
		else
		{
                    (void)printf("non-zero");
		}
		(void)printf(" data detected, pass %d\n", loop);
	    }
	}

        /* read Time data */
	if ( read_time )
	{
	    for ( i = 0; i < in_chans; i++ )
	    {
                CHECK(e1432_read_float64_data(hw, in_chan_list[i],
		  E1432_TIME_DATA, time_data, blocksize, &trailer, &count));
		if ( print_time && i == 0 && ! frame_test )
		{
		    printf("time[%d] = %g, %g, %g,..., %g, %g\n",
		      in_chan_list[i],
		      time_data[0], time_data[1], time_data[2],
		      time_data[blocksize-2], time_data[blocksize-1]);
                    if ( do_peak_rms_print )
		    {
                        FLOATSIZ64 rms = 0;
                        FLOATSIZ64 peak = 0;
			FLOATSIZ64 neg_peak = 1.e100;
			FLOATSIZ64 pos_peak = -1.e100;
                        FLOATSIZ64 dp;
			for ( j = 0; j < count; j++ )
			{
			    dp = time_data[j];
			    if ( dp < neg_peak ) neg_peak = dp; /* neg peak */
			    if ( dp > pos_peak ) pos_peak = dp; /* pos peak */
			    rms += dp * dp;
			}
			rms = sqrt(rms / (FLOATSIZ64)count);
			peak = (pos_peak > -neg_peak) ? pos_peak : -neg_peak;
			printf("time[%d] Peak = %g, Pk/Pk = %g, RMS = %g\n",
			  in_chan_list[i], peak, pos_peak - neg_peak, rms);
		    }
		}
		if ( in_chan_list[i] == plot_chan )
		{
		    for ( j = 0; j < count; j++ ) plot_data[j] = time_data[j];
                    xplot_data_update(plotid);
		}
	    }
	}

        /* read Freq data */
	if ( read_freq )
	{
            if ( num_averages <= 0 )
	    {
		printf("freq: ");
	    }
	    for ( i = 0; i < in_chans; i++ )
	    {
                CHECK(e1432_read_float64_data(hw, in_chan_list[i],
		  E1432_FREQ_DATA, freq_data, blocksize, &trailer, &count));
                freq_raw = .5 * (*rePtr * *rePtr + *imPtr * *imPtr);
	        if ( num_averages <= 0 )
	        {
		    if ( i > 0 && i % 4 == 0 ) printf("\n      ");
		    printf(" %.3f", in_chan_list[i],
		      10 * log10(freq_raw));
	        }
		else
		{
		    freq_avg[i] += freq_raw;
		}
	    }
            if ( num_averages <= 0 )
	    {
		printf("\n");
	    }
	    else if ( loop % num_averages == num_averages - 1 )
	    {
	        /* end of an average */
		printf("freq:\n");
	        for ( i = 0; i < in_chans; i++ )
	        {
		    if ( i > 0 && i % 8 == 0 ) printf("\n      ");
		    printf(" %.3f",
		      10 * log10(freq_avg[i]/(double)num_averages));
	        }
		printf("\n");
	    }
	}

        /* read Octave data */
        overload = 0; /* accumulate overloads */
        prev_gap = 0;
	for ( i = 0; i < in_chans; i++ )
	{
            if ( get_raw )
            {
	        CHECK(e1432_read_raw_data(hw, in_chan_list[i], 
		  E1432_OCTAVE_DATA, octave_raw_data,
		  data_mult * octave_blocksize, &trailer, &count));
	        if ( i == 0 )  /* just print 1 channel */
	        {
                    for ( j = 0; j < count; j += 2 )
		    {
		        printf(" 0x%8.8x 0x%8.8x",
		          octave_raw_data[j], octave_raw_data[j+1]);
                        if ( debug_mode & E1432_OCTAVE_DEBUG_NO_CONVERT )
			{
		            printf("\n");
			}
			else
			{
                            FLOATSIZ32 *fp1 =
			      (FLOATSIZ32 *)&octave_raw_data[j];
                            FLOATSIZ32 *fp2 =
			      (FLOATSIZ32 *)&octave_raw_data[j+1];
		            printf(" %12g %12g", *fp1, *fp2);
                            if ( *fp1 < MAX_LONG && *fp2 < MAX_LONG )
			    {
				printf(" 0x%8.8x 0x%8.8x\n",
				  (LONGSIZ32)(*fp1 + .5),
				  (LONGSIZ32)(*fp2 + .5));
			    }
			    else printf("\n");
			}
		    }
                    if ( do_peak_rms_print )
                    {
                        CHECK(print_peak_rms(hw, in_chan_list[i], &trailer));
                    }
	        }
            }
	    else
            {
                CHECK(e1432_read_float64_data(hw, in_chan_list[i],
		  E1432_OCTAVE_DATA, oct_data[i], octave_blocksize,
		  &trailer, &count));

                if ( frame_test )
                {
                    long val, prev_val;
                    int disc_print = 0;
                    if ( count != octave_blocksize )
                    {
                        (void) fprintf(stderr, "error: (loop %d, chan %d)"
			  " count 0x%x != blocksize 0x%x\n",
                          loop, in_chan_list[i], count, octave_blocksize);
                        ft_errs++;
                    }
                    if ( i == 0 ) /* frame, update computes, first chan */
                    {
                        if ( lin_avg )
                        {
                            update = loop % (avgs / upds);
                        }
                        else
                        {
                            update = loop;
                        }
                        /* allow for average updates */
                        if ( update == 0 ) frame = 0;
                        frame += upds;
                        if ( frame >= avgs ) frame = avgs;
                    }

                    /* first (and second bin) are Pk/RMS, unaveraged */
		    if ( update == 0
                      && !(data_mode == E1432_CONTINUOUS_MODE && loop != 0))
                    {
                        int remainder, raw_first_val;
                        val = octave_data[i][0] + .5;
                        raw_first_val = val;
                        if ( lin_avg )
                        {
                            raw_first_val -=
                              frame_length * upds * (upds - 1) / 2;
                            raw_first_val /= upds;
                        }
                        remainder = raw_first_val % count;
                        if ( remainder != 0 && val < MAX_LONG)
                        {
                            (void) fprintf(stderr, "framing error: loop %d,"
			      " chan %d, first value 0x%x",
                              loop, in_chan_list[i], raw_first_val);
                            if ( lin_avg )
                            {
                                 (void) fprintf(stderr, " (0x%x / %d)",
                                   val, upds);
                            }
                            (void) fprintf(stderr,
                              ", error = 0x%x\n", remainder);
                            ft_errs++;
                        }
                        prev_val = val;
                        prev_first_raw_val = raw_first_val;
                    }
                    else
                    {
                        val = octave_data[i][0] + .5;
                        if ( i == 0 ) /* expected val compute, first chan */
                        {
                            if ( update == 0
                              && data_mode == E1432_CONTINUOUS_MODE)
                            {
                                prev_first_raw_val += frame_length * avgs;
                            }
                            else if ( ! lin_avg ) /* EXP avg */
                            {
                                prev_first_raw_val += frame_length * upds;
                            }
                        }
                        if ( lin_avg )
                        {
                            prev_val = frame * (prev_first_raw_val
                              + frame_length * (frame - 1) / 2);
                        }
                        else
                        {
                            prev_val = prev_first_raw_val;
                        }
                        if ( val != prev_val )
                        {
                            (void) fprintf(stderr,
                              "update frame advance error on loop %d, chan %d:"
                              " expected 0x%x, found 0x%x\n",
                              loop, in_chan_list[i], prev_val, val);
                            ft_errs++;
                            prev_val = val;
                        }
                    }
                    /* the remaining bins, first one has been tested */
                    /* 39 bands + peak and rms bins */
                    for ( j = 1; j < 41 && ft_errs < max_ft_errs; j++ )
                    {
                        if ( lin_avg )
                        {
                            prev_val += frame;
                        }
                        else
                        {
                            prev_val++;
                        }
                        val = octave_data[i][j] + .5;
                        /* don't test if prev_val overflowed (negative) */
                        if ( val != prev_val && prev_val > 0 )
                        {
                            if ( ! disc_print )
                            {
                                (void) fprintf(stderr, "discontinuity error(s)"
				  " loop %d, chan %d, update %d:\n",
				  loop, in_chan_list[i], update);
                                disc_print = 1;
                            }
                            (void) fprintf(stderr,
                              "  at offset %.3d was 0x%6.6x"
                              " (0x%x 0x%x), expected 0x%6.6x\n",
                              j, val, *((int *)&octave_data[i][j] + 1),
                              *(int *)&octave_data[i][j], prev_val);
                            prev_val = val;
                            ft_errs++;
                        }
                    }
                }

                if ( print_oct && (i == 0
		  || spec_chans > 0 || cycle_chans > 0) )
                {
#ifdef	PRINT_DB
		    double tmp;
		    if ( spec_chans > 0 || cycle_chans > 0 )
		    {
		        printf("chan %d:", in_chan_list[i]);
		    }
                    for ( j = 0; j < count; j ++ )
                    {
			if ( j % 6 == 0 ) printf("\n  ");
			tmp = octave_data[i][j];
			if ( tmp > 0 )
			{
			    tmp = 10 * log10(tmp);
			    printf("%9.3f", tmp);
			}
			else
			{
			    printf(" ----.---");
			}
		    }
		    printf("\n");
#else	/* PRINT_DB */
		    if ( spec_chans > 0 || cycle_chans > 0 )
		    {
		        printf("chan %d:\n", in_chan_list[i]);
		    }
                    for ( j = 0; j < count; j += 4 )
                    {
                        printf("  %12.6g  %12.6g  %12.6g  %12.6g\n",
                          octave_data[i][j], octave_data[i][j+1],
                          octave_data[i][j+2], octave_data[i][j+3]);
                    }
#endif	/* PRINT_DB */
                    if ( do_peak_rms_print )
                    {
                        CHECK(print_peak_rms(hw, in_chan_list[i],
                          &trailer));
                    }
                    printf("\n"); /* separate scans */
                }
		if ( num_averages > 0 )
		{
                    for ( j = 0; j < count; j++)
                    {
                        octave_avg[i][j] += octave_data[i][j];
		    }
	            if ( loop % num_averages == num_averages - 1 )
		    {
			/* end of an average */
	                printf("Octave chan %d:\n", in_chan_list[i]);
                        for ( j = 0; j < count; j++ )
                        {
                            if ( j > 0 && j % 6 == 0 ) printf("\n");
                            dTmp = octave_avg[i][j]/(double)num_averages;
			    if ( dTmp > 0 )
			    {
                                printf("  %8.3f", 10 * log10(dTmp));
			    }
			    else
			    {
				printf("  ----.---");
			    }
                        }
                        printf("\n");
		    }
		}

                if ( regression_test && reg_errs < reg_errs_max )
		{
                    FLOATSIZ64 data, expected;
                    FLOATSIZ64 *band;
                    FLOATSIZ64 tmp;
	            low_band_ignore = band_settle_factor / elapsed_time;
		    band = oct_table;
		    tmp = octave_start_freq / half_bandwidth;
		    while ( *band < tmp) band++;
		    for ( j = 0; j < count; j++ )
		    {
			data = octave_data[i][j];
			/* full scale expected, in mean square */
                        expected = .5 * reg_sine_amp * reg_sine_amp;
		        if ( reg_sig_chan[i] )
		        {
		            /* expect a signal in this channel */
			    /* test only if settled */
			    if ( *band > low_band_ignore )
			    {
			        FLOATSIZ64 band_min = *band / half_bandwidth;
			        FLOATSIZ64 band_max = *band * half_bandwidth;
			        FLOATSIZ64 flat_min = *band / flat_bandwidth;
			        FLOATSIZ64 flat_max = *band * flat_bandwidth;
			        FLOATSIZ64 trans_min = *band / trans_bandwidth;
			        FLOATSIZ64 trans_max = *band * trans_bandwidth;
			        if ( reg_sine_freq > flat_min
			          && reg_sine_freq < flat_max )
			        /* in band, in flat part */
			        {
				    if ( data > in_band_max * expected
				      || data < in_band_min * expected )
				    {
				        fprintf(stderr, reg_meas_err,
				          in_chan_list[i], NULL,
				          *band, " (in band)", dB(data));
				        fprintf(stderr,
					  " expected %lg dB to %lg dB\n",
				          dB(in_band_min * expected),
				          dB(in_band_max * expected));
				        reg_errs++;
				    }
			        }
			        if ( reg_sine_freq > band_min
			          && reg_sine_freq < band_max )
			        /* in band on edge */
			        {
				    if ( data > in_band_max * expected
				      || data < band_edge_min * expected )
				    {
				        fprintf(stderr, reg_meas_err,
				          in_chan_list[i], NULL,
				          *band, " (in band, edge)", dB(data));
				        fprintf(stderr,
					  " expected %lg dB to %lg dB\n",
				          dB(band_edge_min * expected),
				          dB(in_band_max * expected));
				        reg_errs++;
				    }
			        }
			        else if ( reg_sine_freq < trans_min
			          || reg_sine_freq > trans_max )
			        /* out of band don't test transition bands */
			        {
			            expected *= out_of_band;
			            if ( data > expected )
			            {
				        fprintf(stderr, reg_meas_err,
				          in_chan_list[i], NULL,
				          *band, " (off band)", dB(data));
				        fprintf(stderr, " expected < %lg dB\n",
				          dB(expected));
				        reg_errs++;
			            }
			        }
                            }
		        }
		        else if ( *band > low_band_ignore )
		        /* expect no signal in this channel */
                        /* ignore unsettled bands */
		        {
			    /* at least 80 dB down */
			    expected *= off_chan_down;
			    if ( data > expected )
			    {
			        fprintf(stderr, reg_meas_err,
			          in_chan_list[i], " (off)",
			          *band, NULL, dB(data));
				fprintf(stderr, " expected < %lg db\n",
				  dB(expected));
				reg_errs++;
			    }
			}
		        band++;
		    }
		}
            }
            if ( print_ovld )
	    {
                if ( trailer.info & E1432_TRAILER_INFO_OVERLOAD )
	        {
		    ovld_tlr_any[i] = 1;
                    overload = 1;
	        }
		else
		{
		    ovld_tlr_any[i] = 0;
		}
                if ( trailer.info & E1432_TRAILER_INFO_OVERLOAD_COMM )
	        {
		    ovld_tlr_com[i] = 1;
                    overload = 1;
	        }
		else
		{
		    ovld_tlr_com[i] = 0;
		}
                ovld_chk_any[i] = 0;  /* e1432_check_overloads() apparently */
                ovld_chk_com[i] = 0;  /* doesn't clear these if set */
                ovld_chk_dif[i] = 0;  /* previously */
                CHECK(e1432_check_overloads(hw, in_chan_list[i],
                  &ovld_chk_any[i], &ovld_chk_com[i], &ovld_chk_dif[i],
		  &ovld_chk_half[i]));
                if ( ovld_chk_any[i] )
		{
                    overload = 1;
		}
	    }
            (void) fflush(NULL);
            if ( read_gap )
	    {
		if ( ((trailer.gap != xpect_gap) || verbose)
		  && trailer.gap != prev_gap )
		{
		    printf("    gap = %d", trailer.gap);
		    if ( trailer.gap != xpect_gap )
		    {
		        printf(" (ERROR, should be %.2f * %.6f = %d)",
                          clock_freq, update_rate, xpect_gap);
		    }
		    printf("\n");
		}
                prev_gap = trailer.gap;
	    }
        }
        if ( read_freq )
	{
	    if ( num_averages > 0 )
	    {
	        if ( loop % num_averages == num_averages - 1 )
		{
                    printf("Oct avg at freq %.2f: ", test_freq);
                    for ( i = 0; i < in_chans; i++ )
	            {
                        dTmp =
			  octave_avg[i][test_freq_index]/(double)num_averages;
	                if ( i > 0 && i % 4 == 0 ) printf("\n      ");
	                printf(" %.3f", 10 * log10(dTmp));
	            }
	            printf("\n");
		    }
            }
	    else
	    {
                printf("Oct at freq %.2f: ", test_freq);
                for ( i = 0; i < in_chans; i++ )
	        {
                    dTmp = octave_data[i][test_freq_index]/(double)num_averages;
	            if ( i > 0 && i % 4 == 0 ) printf("\n      ");
	            printf(" %.3f", 10 * log10(dTmp));
	        }
	        printf("\n");
	    }
	}
        if ( overload )
	{
	    printf("overloads (channel order");
	    for ( i = 0; i < in_chans; i++ ) printf(" %d,", in_chan_list[i]);
	    printf(")\n  trailer any ");
	    for ( i = 0; i < in_chans; i++ ) printf("%d", ovld_tlr_any[i]);
	    printf(", comm ");
	    for ( i = 0; i < in_chans; i++ ) printf("%d", ovld_tlr_com[i]);
	    printf("\n  check_overloads() any ");
	    for ( i = 0; i < in_chans; i++ ) printf("%d", ovld_chk_any[i]);
	    printf(", comm ");
	    for ( i = 0; i < in_chans; i++ ) printf("%d", ovld_chk_com[i]);
	    printf(", diff ");
	    for ( i = 0; i < in_chans; i++ ) printf("%d", ovld_chk_dif[i]);
	    printf("\n");
	}

	if ( read_current )
	{
            CHECK(e1432_get_current_data(hw, in_group, E1432_OCTAVE_DATA,
	      E1432_DATA_SIZE_FLOAT64, (void *)oct_data, &count));
	    if ( frame_test )
	    {
		int errs = 0;
		long val = *oct_data[0] + .5;
		long xpect;
		if ( val % count != 0 )
		{
		    (void) fprintf(stderr, "current framing error, loop %d"
		      " chan %d = 0x%x\n", loop, in_chan_list[i], val);
		    errs++;
                    ft_errs++;
		}
		else
		{
		    xpect = val;
                    for ( i = 1; i < in_chans; i++ )
		    {
		        val = *oct_data[i] + .5;
		        if ( val != xpect )
		        {
		            (void) fprintf(stderr, "current framing error"
		              " loop %d, chan %d = 0x%x, expected 0x%x\n",
		              loop, in_chan_list[i], val, xpect);
		            errs++;
                            ft_errs++;
		        }
		    }
		}
		if ( errs <= 0 )
		{
		    int last_bin = 41;  /* if no lin > 1 averaging */
		    long n_avgs = 1;   /* unless proven otherwise */
		    if ( lin_avg && avgs > 1 ) last_bin = 2;

		    if ( lin_avg && avgs > 1 )
		    {
			/* increment between bins should be n_avgs */
		        val = *(oct_data[0] + 1) + .5;
                        n_avgs = val - xpect;
			if ( n_avgs < 1 || n_avgs > avgs )
			{
	                    (void) fprintf(stderr, "current avg comput err"
			      " loop %d, 0x%x - 0x%x = 0x%x = %d avgs\n",
			      loop, val, xpect, n_avgs, n_avgs);
		            errs++;
                            ft_errs++;
			}
		    }
		    for ( j = 1; j < last_bin && errs <= 0; j++ )
		    {
                        xpect += n_avgs;
                        for ( i = 0; i < in_chans; i++ )
		        {
			    val = *(oct_data[i] + j) + .5;
			    if ( val != xpect )
			    {
		                (void) fprintf(stderr,
				  "current discontinuity, loop %d, chan %d"
				  " offset %d = 0x%x, expected 0x%x\n",
			          loop, in_chan_list[i], j, val, xpect);
			        errs++;
                                ft_errs++;
			        break;
			    }
		        }
		    }
		}
	    }
	    if ( print_current )
	    {
                for ( i = 0; i < in_chans; i++ )
                {
		    int last_bin = 41;
		    if ( last_bin > count ) last_bin = count;
                    printf("ch %d %g %g %g %g ... %g %g\n",
		      in_chan_list[i],
                      *oct_data[i],
		      *(oct_data[i]+1),
		      *(oct_data[i]+2),
		      *(oct_data[i]+3),
		      *(oct_data[i]+last_bin-2),
		      *(oct_data[i]+last_bin-1));
                }
	    }
	}
        if ( print_oct ) printf("\n"); /* separate scans */

        if ( debug_pause )
        {
            prompt("hit enter to continue...");
        }

        if ( meas_restart > 0 && loop % meas_restart == (meas_restart-1)
	  && loop != loops - 1 )
        {
	    printf("restarting measurement");
	    if ( cycle_chans > 0 )  /* more complete stop/restart */
	    {
	        /* end Oct meas stuff */
                CHECK(e1432_finish_measure(hw, in_group));
                CHECK(e1432_set_octave_meas(hw, in_group,
		  E1432_OCTAVE_MEAS_OFF));
                CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_OCTAVE,
                  E1432_ENABLE_OFF));
                CHECK(e1432_set_active(hw, in_group, E1432_CHANNEL_OFF));
	        CHECK(e1432_delete_channel_group(hw, in_group));
	        in_chans = 1; /* only 1 channel from now on */

		/* do some FFT measurements if requested */
		if ( fft_meas > 0 )
		{
		    int f;
		    SHORTSIZ16 chan;

		    chan = cycle_chan_list[cycle_chan];
	            in_chan_list[0] = chan;
	            printf(", fft, chan = %d\n", chan);
	            if ( ++cycle_chan >= cycle_chans ) cycle_chan = 0;
                    in_group =
		      e1432_create_channel_group(hw, in_chans, in_chan_list);
                    if (in_group >= 0)
                    {
                        DEBUG((void) printf("e1432_create_channel_group"
			  " in_group returned %d\n", in_group));
                        return -1;
                    }
                    /* not really needed */
                    CHECK(e1432_set_calc_data(hw, in_group, E1432_DATA_FREQ));
                    CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_FREQ,
                      E1432_ENABLE_ON));
                    CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_TIME,
                      E1432_ENABLE_OFF));

                    /* Start measurement */
                    CHECK(e1432_init_measure(hw, in_group));

		    for ( f = 0; f < fft_meas; f++ )
		    {
	                if ( wait_block_available(hw, in_group, loop) < 0 )
	                {
                            ft_errs++;
	                    break;
	                }
			/* read the one and only channel of data */
                        CHECK(e1432_read_float64_data(hw, chan, E1432_FREQ_DATA,
			  freq_data, blocksize, &trailer, &count));
			/* look over the freq data */
			{
			    FLOATSIZ64 ms, min_f, max_f;
			    FLOATSIZ64 *fp = &freq_data[0];
			    FLOATSIZ64 min = 1.e100;
			    FLOATSIZ64 max = 0.0;
	                    for ( i = 0; i < blocksize; i += 2 )
			    {
				ms = *fp * (*fp++);
				ms += *fp * (*fp++);
				if ( ms > max )
				{
				    max = ms;
				    max_f = i/2 * bin_space;
				}
				else if ( ms < min )
				{
				    min = ms;
				    min_f = i/2 * bin_space;
				}
			    }
			    printf(" chan %d FFT max = %.3f @ %.3f, min= %.3f"
			      " @ %.3f\n",
			      chan, dB(max), max_f, dB(min), min_f);
			}
                    }

                    /* shut down FFT meas */
                    CHECK(e1432_finish_measure(hw, in_group));
                    /* not really needed */
                    CHECK(e1432_set_calc_data(hw, in_group, E1432_DATA_OCTAVE));
                    CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_FREQ,
                      E1432_ENABLE_OFF));
                    CHECK(e1432_set_active(hw, in_group, E1432_CHANNEL_OFF));
	            CHECK(e1432_delete_channel_group(hw, in_group));

	            printf("restarting measurement");
		}

                /* new Oct meas stuff */
	        in_chan_list[0] = cycle_chan_list[cycle_chan];
	        printf(", chan = %d", cycle_chan_list[cycle_chan]);
	        if ( ++cycle_chan >= cycle_chans ) cycle_chan = 0;
                in_group =
		  e1432_create_channel_group(hw, in_chans, in_chan_list);
                if (in_group >= 0)
                {
                    DEBUG((void) printf("e1432_create_channel_group in_group"
	              " returned %d\n", in_group));
                    return -1;
                }
                CHECK(e1432_set_octave_meas(hw, in_group,
		  E1432_OCTAVE_MEAS_ON));
                CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_OCTAVE,
                  E1432_ENABLE_ON));
                CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_TIME,
                  E1432_ENABLE_OFF));
            }
            printf("\n");
            CHECK(e1432_init_measure(hw, in_group));
	}

        elapsed_time += update_rate;

	if ( num_averages > 0 && loop % num_averages == num_averages - 1 )
	{
            printf("\n"); /* separate averages */
            zero_avg_buffs();
	}
    }

    /* print out test conditions if a frame test failure occurred */
    if ( ft_errs )
    {
	int k;
	char *a, *n, *F, *M, *V;
	char e[20];

        F = frame_test ? "-F " : "";
	n = warnings ? "" : "-n ";
        for ( k = 0; data_mode_enums[k] != data_mode ; k++ );
        M = data_mode_strs[k];
        for ( k = 0; octave_avg_mode_enums[k] != octave_avg_mode ; k++ );
        a = octave_avg_mode_strs[k];
	V = read_current ? " -V" : "";
	e[0] = NULL;
	if ( max_ft_errs < MAX_LONG ) sprintf(e, " -e%d", max_ft_errs);

	(void) fprintf(stderr,
	  "options: %s%s-N%d -M%s -a%s -t%.3f, -i%.3f%s%s\n\n",
	  F, n, loops, M, a, octave_time_step, octave_int_time, V, e);
        (void) fflush(NULL);
    }

    return 0;
}
